#!/usr/bin/env python3
"""
ANTHROHEART FINAL POLISH SCRIPT
Author: Thomas B. Sweet AKA Anthro Teacher (Cio)
License: Public Domain (CC0)

Recursive, archival-quality branding + timestamping.
Run from the ROOT of your project.
"""

import os
import io
import time
import hashlib
import datetime
import requests

from PIL import Image, ImageDraw, ImageFont
from mutagen.mp3 import MP3
from mutagen.id3 import (
    ID3, APIC, TIT2, TPE1, TALB, TDRC, TCOP, COMM
)
from mutagen.wave import WAVE
from mutagen import File

# ================= CONFIG =================

ARTIST = "Thomas B. Sweet AKA Anthro Teacher (Cio)"
ALBUM = "The AnthroHeart Collection"
YEAR = "2025"
LICENSE = "Public Domain (CC0)"
WEBSITE = "https://anthroentertainment.com/"
DONATE = "donate.anthroentertainment.com"
AUDIO_ICON_URL = "http://dogicon.anthroentertainment.com/"

IMAGE_HEADER_TEXT = (
    "Public Domain (CC0) Gift\n"
    "https://anthroentertainment.com/\n"
    "donate.anthroentertainment.com\n\n"
    "Audio Icon: http://dogicon.anthroentertainment.com/"
)

# =========================================

def archival_timestamp():
    utc = datetime.datetime.now(datetime.timezone.utc).isoformat()
    epoch = str(int(time.time()))
    block = f"{utc}|{epoch}"
    sha = hashlib.sha256(block.encode("utf-8")).hexdigest()
    return utc, epoch, sha

def get_audio_icon():
    r = requests.get(AUDIO_ICON_URL, timeout=20)
    r.raise_for_status()
    return r.content

AUDIO_ICON_BYTES = get_audio_icon()

def process_image(path):
    img = Image.open(path).convert("RGBA")
    width, height = img.size

    header_height = max(80, height // 6)
    new_img = Image.new("RGBA", (width, height + header_height), (255, 255, 255, 255))
    new_img.paste(img, (0, header_height))

    draw = ImageDraw.Draw(new_img)
    try:
        font = ImageFont.truetype("arial.ttf", 20)
    except:
        font = ImageFont.load_default()

    draw.multiline_text(
        (10, 10),
        IMAGE_HEADER_TEXT,
        fill=(0, 0, 0),
        font=font,
        spacing=4
    )

    new_img.convert("RGB").save(path)
    print(f"🖼️ Image stamped: {path}")

def process_wav(path):
    audio = WAVE(path)
    utc, epoch, sha = archival_timestamp()

    title = os.path.splitext(os.path.basename(path))[0]

    # WAV-safe RIFF INFO fields
    audio["INAM"] = title
    audio["IART"] = ARTIST
    audio["IPRD"] = ALBUM
    audio["ICRD"] = YEAR
    audio["ICOP"] = LICENSE
    audio["IURL"] = WEBSITE

    # Album art is a HYPERLINK stored in metadata (archival-safe)
    audio["ICMT"] = (
        "Public Domain (CC0) Gift\n"
        f"{WEBSITE}\n"
        f"{DONATE}\n\n"
        "Album Art (URL):\n"
        f"{AUDIO_ICON_URL}\n\n"
        f"Archival UTC: {utc}\n"
        f"Epoch: {epoch}\n"
        f"SHA256: {sha}"
    )

    audio.save()
    print(f"🔊 WAV tagged (album art = hyperlink): {path}")

def process_mp3(path):
    audio = MP3(path, ID3=ID3)
    try:
        audio.add_tags()
    except:
        pass

    utc, epoch, sha = archival_timestamp()
    title = os.path.splitext(os.path.basename(path))[0]

    audio.tags.add(TIT2(encoding=3, text=title))
    audio.tags.add(TPE1(encoding=3, text=ARTIST))
    audio.tags.add(TALB(encoding=3, text=ALBUM))
    audio.tags.add(TDRC(encoding=3, text=YEAR))
    audio.tags.add(TCOP(encoding=3, text=LICENSE))
    audio.tags.add(COMM(
        encoding=3,
        lang="eng",
        desc="Archival",
        text=(
            f"Public Domain (CC0) Gift\n"
            f"{WEBSITE}\n{DONATE}\n"
            f"Archival UTC: {utc}\nEpoch: {epoch}\nSHA256: {sha}"
        )
    ))

    audio.tags.add(APIC(
        encoding=3,
        mime="image/png",
        type=3,
        desc="Front Cover",
        data=AUDIO_ICON_BYTES
    ))

    audio.save()
    print(f"🎵 MP3 tagged + album art: {path}")

def process_txt(path):
    utc, epoch, sha = archival_timestamp()

    header = (
        f"ARCHIVAL TIMESTAMP\n"
        f"UTC: {utc}\n"
        f"Epoch: {epoch}\n"
        f"SHA256: {sha}\n\n"
        f"{WEBSITE}\n"
        f"{DONATE}\n\n"
    )

    with open(path, "r", encoding="utf-8", errors="ignore") as f:
        content = f.read()

    with open(path, "w", encoding="utf-8") as f:
        f.write(header + content)

    print(f"📄 TXT stamped: {path}")

def main():
    for root, _, files in os.walk("."):
        for name in files:
            path = os.path.join(root, name)
            lower = name.lower()

            try:
                if lower.endswith((".png", ".jpg", ".jpeg")):
                    process_image(path)
                elif lower.endswith(".wav"):
                    process_wav(path)
                elif lower.endswith(".mp3"):
                    process_mp3(path)
                elif lower.endswith(".txt"):
                    process_txt(path)
            except Exception as e:
                print(f"⚠️ Failed: {path} — {e}")

if __name__ == "__main__":
    main()
